import pygame


class CharacterStatusWindow:
    def __init__(self, player, font=None):
        self.player = player
        self.font = font or pygame.font.SysFont("Segoe UI", 22)
        self.visible = True

        self.display_map = [
            ("Level", "Level"),
            ("HP", "HP"),
            ("MP", "MP"),
            ("", ""),
            ("", "— Attributes —"),
            ("STR", "Strength"),
            ("DEX", "Dexterity"),
            ("INT", "Intelligence"),
            ("END", "Endurance"),
            ("VIT", "Vitality"),
            ("", ""),
            ("", "— Combat —"),
            ("ATT", "Attack"),
            ("ATT_SPD", "Attack Speed"),
            ("DEF", "Defense"),
            ("SPD", "Speed"),
        ]

    def _get_item_bonus(self, stat_name):
        """Sums all bonuses from equipped items for a given stat."""
        total_bonus = 0
        if not hasattr(self.player, "equipment"):
            return 0

        for slot, item in self.player.equipment.items():
            if not item:
                continue
            bonuses = item.get("bonuses", {})
            total_bonus += bonuses.get(stat_name, 0)
        return total_bonus

    def draw(self, screen, pos=(50, 50)):
        if not self.visible:
            return

        x, y = pos
        line_spacing = 26
        padding = 12
        label_x = x + 10

        visible_lines = len(self.display_map)
        bg_height = padding * 2 + line_spacing * visible_lines
        bg_width = 340
        bg_right_edge = x + bg_width

        # Background
        surface = pygame.Surface((bg_width, bg_height), pygame.SRCALPHA)
        surface.fill((10, 10, 30, 180))
        pygame.draw.rect(surface, (120, 150, 255), surface.get_rect(), 2)
        screen.blit(surface, (x, y))

        for i, (key, label) in enumerate(self.display_map):
            line_y = y + padding + i * line_spacing

            # Blank lines
            if key == "" and label == "":
                continue

            # Section header
            if key == "" and label.startswith("—"):
                header_surf = self.font.render(label, True, (180, 210, 255))
                screen.blit(header_surf, (label_x, line_y))
                continue

            base = self.player.stats.get(key, 0) if hasattr(self.player, "stats") else 0
            bonus = self._get_item_bonus(key)
            total = base + bonus

            # Render stat label with colon
            label_surf = self.font.render(f"{label}:", True, (255, 255, 255))
            screen.blit(label_surf, (label_x, line_y))

            # Render total value next to label
            total_surf = self.font.render(f" {total}", True, (200, 230, 255))
            total_x = label_x + label_surf.get_width()
            screen.blit(total_surf, (total_x, line_y))

            # Render breakdown (base + bonus) right aligned to background edge with padding
            breakdown_text = f"({base} + {bonus})"
            breakdown_surf = self.font.render(breakdown_text, True, (180, 255, 180))
            breakdown_x = bg_right_edge - padding - breakdown_surf.get_width()
            screen.blit(breakdown_surf, (breakdown_x, line_y))
