# client/src/portal.py
import pygame
import math
import numpy as np

class Portal:
    def __init__(self, width, height):
        self.surface = pygame.Surface((width, height), pygame.SRCALPHA)
        self.time = 0

    def draw(self, surface, x, y, radius=100, dt=0.016, alpha=255):
        self.time += dt

        # Base colors
        gray_center = int(20 + 10 * (0.5 + 0.5 * math.sin(self.time * 1.5)))  # oscillating gray
        color_center = (gray_center, gray_center, gray_center)
        color_edge = (0, 0, 0)

        # Create gradient
        portal_surf = pygame.Surface((radius*2, radius*2), pygame.SRCALPHA)
        for i in range(radius, 0, -1):
            t = i / radius
            r = int(color_edge[0] * (1-t) + color_center[0] * t)
            g = int(color_edge[1] * (1-t) + color_center[1] * t)
            b = int(color_edge[2] * (1-t) + color_center[2] * t)
            a = int(alpha * t)
            pygame.draw.circle(portal_surf, (r, g, b, a), (radius, radius), i)

        # Rotate slightly to simulate left rotation
        angle = self.time * 30  # degrees per second
        rotated = pygame.transform.rotate(portal_surf, angle)

        # Center the rotated portal on x, y
        rect = rotated.get_rect(center=(x, y))
        surface.blit(rotated, rect.topleft)


class PortalManager:
    def __init__(self, portal_renderer, portals):
        self.renderer = portal_renderer
        self.portals = portals
        self.timer = 0.0

    def update(self, dt):
        self.timer += dt

    def draw(self, surface, cam_x, cam_y):
        FADE_IN_START = 5
        FADE_IN_DURATION = 10
        if self.timer < FADE_IN_START:
            return
        elapsed = self.timer - FADE_IN_START
        alpha = min(255, int(255 * (elapsed / FADE_IN_DURATION)))
        for portal_data in self.portals:
            self.renderer.draw(surface,
                               portal_data["x"] - cam_x,
                               portal_data["y"] - cam_y,
                               radius=int(portal_data["radius"] * 300),
                               dt=0,  # pass your clock delta
                               alpha=alpha)
