# client/src/combat_effects.py

import pygame
import random

class DamageNumber:
    """Floating damage numbers that appear when damage is dealt."""
    
    def __init__(self, x, y, damage, is_crit=False, is_player=False):
        self.x = x
        self.y = y
        self.damage = damage
        self.is_crit = is_crit
        self.is_player = is_player
        
        # Visual properties
        self.lifetime = 1.5  # seconds
        self.age = 0.0
        self.velocity_y = -50  # float upward
        self.velocity_x = random.uniform(-10, 10)  # slight random horizontal drift
        
        # Color based on type
        if is_player:
            self.color = (255, 100, 100)  # Red for damage to player
        elif is_crit:
            self.color = (255, 255, 0)  # Yellow for crits
        else:
            self.color = (255, 255, 255)  # White for normal damage
        
        # Font size
        self.font_size = 36 if is_crit else 32
        self.font = pygame.font.Font(None, self.font_size)
    
    def update(self, dt):
        """Update position and age."""
        self.age += dt
        self.y += self.velocity_y * dt
        self.x += self.velocity_x * dt
        
        # Slow down over time
        self.velocity_y *= 0.95
        self.velocity_x *= 0.95
        
        return self.age < self.lifetime
    
    def draw(self, surface, camera_x=0, camera_y=0):
        """Draw the damage number."""
        # Calculate alpha based on age (fade out)
        alpha = int(255 * (1 - self.age / self.lifetime))
        
        # Render text
        text = f"-{int(self.damage)}" if self.is_player else str(int(self.damage))
        if self.is_crit:
            text = f"{text}!"
        
        text_surface = self.font.render(text, True, self.color)
        text_surface.set_alpha(alpha)
        
        # Draw with camera offset
        draw_x = self.x - camera_x - text_surface.get_width() // 2
        draw_y = self.y - camera_y
        
        surface.blit(text_surface, (draw_x, draw_y))


class HitEffect:
    """Flash effect when something is hit."""
    
    def __init__(self, x, y, radius=60):
        self.x = x
        self.y = y
        self.radius = radius
        self.max_radius = radius
        self.lifetime = 0.3
        self.age = 0.0
    
    def update(self, dt):
        """Expand and fade."""
        self.age += dt
        self.radius = self.max_radius * (1 + self.age / self.lifetime)
        return self.age < self.lifetime
    
    def draw(self, surface, camera_x=0, camera_y=0):
        """Draw expanding circle."""
        alpha = int(255 * (1 - self.age / self.lifetime))
        color = (255, 200, 0, alpha)
        
        draw_x = int(self.x - camera_x)
        draw_y = int(self.y - camera_y)
        
        # Draw semi-transparent circle
        s = pygame.Surface((int(self.radius * 2), int(self.radius * 2)), pygame.SRCALPHA)
        pygame.draw.circle(s, color, (int(self.radius), int(self.radius)), int(self.radius), 3)
        surface.blit(s, (draw_x - self.radius, draw_y - self.radius))